<?php
/**
 * Handle the Premium nutrition label shortcode.
 *
 * @link       http://bootstrapped.ventures
 * @since      5.6.0
 *
 * @package    WP_Recipe_Maker_Premium
 * @subpackage WP_Recipe_Maker_Premium/includes/public/shortcodes/recipe
 */

/**
 * Handle the Premium nutrition label shortcode.
 *
 * @since      5.6.0
 * @package    WP_Recipe_Maker_Premium
 * @subpackage WP_Recipe_Maker_Premium/includes/public/shortcodes/recipe
 * @author     Brecht Vandersmissen <brecht@bootstrapped.ventures>
 */
class WPRMP_SC_Nutrition_Label {
	public static function init() {
		add_filter( 'wprm_nutrition_label_shortcode', array( __CLASS__, 'shortcode' ), 10, 3 );
	}

	/**
	 * Nutrition label shortcode.
	 *
	 * @since	5.6.0
	 * @param	mixed $output Current output.
	 * @param	array $atts   Options passed along with the shortcode.
	 * @param	mixed $recipe Recipe the shortcode is getting output for.
	 */
	public static function shortcode( $output, $atts, $recipe ) {
		$output = '';

		$align = in_array( $atts['align'], array( 'center', 'right' ) ) ? $atts['align'] : 'left';
		$output .= WPRM_Shortcode_Helper::get_section_header( $atts, 'nutrition' );

		// Output.
		$classes = array(
			'wprm-nutrition-label-container',
			'wprm-nutrition-label-container-' . $atts['style'],
		);

		// Add custom class if set.
		if ( $atts['class'] ) { $classes[] = esc_attr( $atts['class'] ); }

		if ( 'label' !== $atts['style'] ) {
			$classes[] = 'wprm-block-text-' . $atts['text_style'];
		}

		$output .= '<div class="' . esc_attr( implode( ' ', $classes ) ) . '" style="text-align: ' . $align . ';">';

		switch ( $atts['style'] ) {
			case 'simple':
			case 'grouped':
				$nutrition = $recipe->nutrition();
				$serving_unit = isset( $nutrition['serving_unit'] ) && $nutrition['serving_unit'] ? $nutrition['serving_unit'] : WPRM_Settings::get( 'nutrition_default_serving_unit' );

				$show_daily = (bool) $atts['daily'];
				$type = in_array( $atts['nutrition_values'], array( 'serving', '100g' ) ) ? $atts['nutrition_values'] : 'serving';

				// Multiply factor defaults to 1.
				$multiply_factor = 1 ;

				// Check if we actually can display per 100g.
				if ( '100g' === $type ) {
					// Serving unit needs to be in grams (allow empty as well).
					if ( ! in_array( $serving_unit, array( '', 'g', __( 'gram', 'wp-recipe-maker' ), __( 'grams', 'wp-recipe-maker' ) ) ) ) {
						return '';
					}

					// Serving size needs to be set to calculate multiply factor.
					$multiply_factor = false;

					$serving_size = isset( $nutrition[ 'serving_size' ] ) && false !== $nutrition[ 'serving_size' ] ? WPRM_Recipe_Parser::parse_quantity( $nutrition[ 'serving_size' ] ) : 0;
					if ( is_numeric( $serving_size ) && 0 < $serving_size ) {
						$multiply_factor = 100.00 / $serving_size;
					}

					// Need to have a factor to multiply by to display per 100g.
					if ( ! $multiply_factor ) {
						return '';
					}
				}

				// Get nutrition output for each field.
				$nutrition_output = array();

				$nutrition_fields = WPRM_Nutrition::get_fields();
				$nutrition_fields['serving_size']['unit'] = $serving_unit;

				foreach ( $nutrition_fields as $field => $options ) {
					if ( isset( $nutrition[ $field ] ) && false !== $nutrition[ $field ] && ( WPRM_Settings::get( 'nutrition_label_zero_values' ) || $nutrition[ $field ] ) ) {

						// Nutrition value.
						$value = $nutrition[ $field ];
						$display_value = $value;

						if ( 1 !== $multiply_factor ) {
							$value = WPRM_Recipe_Parser::parse_quantity( $nutrition[ $field ] ) * $multiply_factor;
							$display_value = WPRM_Recipe_Parser::format_quantity( $value, WPRM_Settings::get( 'nutrition_facts_calculation_round_to_decimals' ) );
						}

						// Calculate percentage if needed.
						$percentage = false;
						if ( $show_daily && isset( $options['daily'] ) && $options['daily'] ) {
							$percentage = round( floatval( $value ) / $options['daily'] * 100 );
						}

						$style = '';
						if ( 'grouped' === $atts['style'] ) {
							$style = 'style="flex-basis: ' . esc_attr( $atts['group_width'] ) . '"';
						}

						$field_output = '<span class="wprm-nutrition-label-text-nutrition-container wprm-nutrition-label-text-nutrition-container-' . esc_attr( $field ) . '"' . $style .'>';
						$field_output .= '<span class="wprm-nutrition-label-text-nutrition-label  wprm-block-text-' . esc_attr( $atts['label_style'] ) . '" style="color: ' . esc_attr( $atts['label_color'] ) . '">' . WPRM_Shortcode_Helper::sanitize_html( __( $options['label'] , 'wp-recipe-maker-premium' ) ) . WPRM_Shortcode_Helper::sanitize_html( $atts['label_separator'] ) . '</span>';
						$field_output .= '<span class="wprm-nutrition-label-text-nutrition-value" style="color: ' . esc_attr( $atts['value_color'] ) . '">' . $display_value . '</span>';
						$field_output .= WPRM_Shortcode_Helper::sanitize_html( $atts['unit_separator'] );
						$field_output .= '<span class="wprm-nutrition-label-text-nutrition-unit" style="color: ' . esc_attr( $atts['value_color'] ) . '">' . $options['unit'] . '</span>';

						if ( $percentage ) {
							$field_output .= '<span class="wprm-nutrition-label-text-nutrition-daily" style="color: ' . esc_attr( $atts['value_color'] ) . '">';

							switch ( $atts['daily_seperator'] ) {
								case 'parentheses':
									$field_output .= ' (' . $percentage . '%)';
									break;
								case 'dash':
									$field_output .= ' - ' . $percentage . '%';
									break;
								default:
									$field_output .= ' ' . $percentage . '%';
							}

							$field_output .= '</span>';
						}

						$field_output .= '</span>';

						$nutrition_output[] = $field_output;
					}
				}

				if ( ! count( $nutrition_output ) ) {
					return '';
				}

				$nutrition_separator = '';
				if ( 'simple' === $atts['style'] ) {
					$nutrition_separator = '<span style="color: ' . esc_attr( $atts['label_color'] ) . '">' . WPRM_Shortcode_Helper::sanitize_html( $atts['nutrition_separator'] ) . '</span>';
				}
				
				$output .= implode( $nutrition_separator, $nutrition_output );
				break;
			default:
				$label = WPRMP_Nutrition::label( $recipe );
				if ( ! $label ) {
					return '';
				}

				if ( 'legacy' === WPRM_Settings::get( 'nutrition_label_style' ) ) {
					$style = 'style="';
					$style .= 'background-color: ' . esc_attr( $atts['label_background_color'] ) . ';';
					$style .= 'color: ' . esc_attr( $atts['label_text_color'] ) . ';';
					$style .= '"';

					$label = str_replace( 'class="wprm-nutrition-label"', 'class="wprm-nutrition-label" ' . $style, $label );
				}
				
				$output .= $label;
			}

		$output .= '</div>';

		return $output;
	}
}

WPRMP_SC_Nutrition_Label::init();